﻿//=============================================================================
// Rain.fx Frank Luna (C) 2011 Wszelkie prawa zastrzeżone.
//
// System cząsteczek deszczu. Cząsteczki emitowane bezpośrednio w przestrzeni świata.
//=============================================================================


//***********************************************
// ZMIENNE GLOBALNE                                     *
//***********************************************

cbuffer cbPerFrame
{
	float3 gEyePosW;
	
	// jeśli punkt i kierunek emisji są różne
	float3 gEmitPosW;
	float3 gEmitDirW;
	
	float gGameTime;
	float gTimeStep;
	float4x4 gViewProj; 
};

cbuffer cbFixed
{
	// Stałe wypadkowe przyspieszenie do przyspieszania cząsteczek.
	float3 gAccelW = {-1.0f, -9.8f, 0.0f};
};
 
// Tablica tekstur do teksturowania cząsteczek.
Texture2DArray gTexArray;

// Losowa tekstura do generowania liczb losowych w shaderach.
Texture1D gRandomTex;
 
SamplerState samLinear
{
	Filter = MIN_MAG_MIP_LINEAR;
	AddressU = WRAP;
	AddressV = WRAP;
};

DepthStencilState DisableDepth
{
    DepthEnable = FALSE;
    DepthWriteMask = ZERO;
};

DepthStencilState NoDepthWrites
{
    DepthEnable = TRUE;
    DepthWriteMask = ZERO;
};


//***********************************************
// FUNKCJE POMOCNICZE                             *
//***********************************************
float3 RandUnitVec3(float offset)
{
	// Użyj czasu gry plus przesunięcia do próbkowania losowej tekstury.
	float u = (gGameTime + offset);
	
	// współrzędne w [–1,1]
	float3 v = gRandomTex.SampleLevel(samLinear, u, 0).xyz;
	
	// rzutuj na sferę jednostkową
	return normalize(v);
}

float3 RandVec3(float offset)
{
	// Użyj czasu gry plus przesunięcia do próbkowania losowej tekstury.
	float u = (gGameTime + offset);
	
	// współrzędne w [–1,1]
	float3 v = gRandomTex.SampleLevel(samLinear, u, 0).xyz;
	
	return v;
}
 
//***********************************************
// TECHNIKA STRUMIENIOWA                              *
//***********************************************

#define PT_EMITTER 0
#define PT_FLARE 1
 
struct Particle
{
	float3 InitialPosW : POSITION;
	float3 InitialVelW : VELOCITY;
	float2 SizeW       : SIZE;
	float Age          : AGE;
	uint Type          : TYPE;
};
  
Particle StreamOutVS(Particle vin)
{
	return vin;
}

// Strumieniowy shader geometrii odpowiada za wytwarzanie nowych cząsteczek i niszczenie starych. 
// Kod techniki będzie inny w zależności od systemu cząsteczek,
// ponieważ inne będą reguły niszczenia, tworzenia i renderowania cząsteczek.
[maxvertexcount(6)]
void StreamOutGS(point Particle gin[1], 
                 inout PointStream<Particle> ptStream)
{	
	gin[0].Age += gTimeStep;

	if( gin[0].Type == PT_EMITTER )
	{	
		// Czas wyemitować nową cząsteczkę?
		if( gin[0].Age > 0.002f )
		{
			for(int i = 0; i < 5; ++i)
			{
				// Rozprzestrzeniaj krople deszczu ponad kamerą.
				float3 vRandom = 35.0f*RandVec3((float)i/5.0f);
				vRandom.y = 20.0f;
			
				Particle p;
				p.InitialPosW = gEmitPosW.xyz + vRandom;
				p.InitialVelW = float3(0.0f, 0.0f, 0.0f);
				p.SizeW       = float2(1.0f, 1.0f);
				p.Age         = 0.0f;
				p.Type        = PT_FLARE;
			
				ptStream.Append(p);
			}
			
			// Wyzeruj czas do emisji.
			gin[0].Age = 0.0f;
		}

		// zawsze zachowuj emitery
		ptStream.Append(gin[0]);
	}
	else
	{
		// Określ warunki, przy których cząsteczka zostanie zachowana. Mogą się różnić w zależności od systemu.
		if( gin[0].Age <= 3.0f )
			ptStream.Append(gin[0]);
	}		
}

GeometryShader gsStreamOut = ConstructGSWithSO( 
	CompileShader( gs_5_0, StreamOutGS() ), 
	"POSITION.xyz; VELOCITY.xyz; SIZE.xy; AGE.x; TYPE.x" );
	
technique11 StreamOutTech
{
    pass P0
    {
        SetVertexShader( CompileShader( vs_5_0, StreamOutVS() ) );
        SetGeometryShader( gsStreamOut );
        
        // wyłącz shader pikseli dla techniki strumieniowej bez renderowania
        SetPixelShader(NULL);
        
        // przy technice strumieniowej bez renderowania trzeba również wyłączyć bufor głębokości
        SetDepthStencilState( DisableDepth, 0 );
    }
}

//***********************************************
// TECHNIKA RYSOWANIA                                    *
//***********************************************

struct VertexOut
{
	float3 PosW  : POSITION;
	uint   Type  : TYPE;
};

VertexOut DrawVS(Particle vin)
{
	VertexOut vout;
	
	float t = vin.Age;
	
	// równanie przyspieszenia stałego
	vout.PosW = 0.5f*t*t*gAccelW + t*vin.InitialVelW + vin.InitialPosW;
	
	vout.Type  = vin.Type;
	
	return vout;
}

struct GeoOut
{
	float4 PosH  : SV_Position;
	float2 Tex   : TEXCOORD;
};

// Rysujący shader geometrii rozszerza punkty do linii.
[maxvertexcount(2)]
void DrawGS(point VertexOut gin[1], 
            inout LineStream<GeoOut> lineStream)
{	
	// Nie rysuj emiterów.
	if( gin[0].Type != PT_EMITTER )
	{
		// Nachyl linię w kierunku przyspieszenia.
		float3 p0 = gin[0].PosW;
		float3 p1 = gin[0].PosW + 0.07f*gAccelW;
		
		GeoOut v0;
		v0.PosH = mul(float4(p0, 1.0f), gViewProj);
		v0.Tex = float2(0.0f, 0.0f);
		lineStream.Append(v0);
		
		GeoOut v1;
		v1.PosH = mul(float4(p1, 1.0f), gViewProj);
		v1.Tex  = float2(1.0f, 1.0f);
		lineStream.Append(v1);
	}
}

float4 DrawPS(GeoOut pin) : SV_TARGET
{
	return gTexArray.Sample(samLinear, float3(pin.Tex, 0));
}

technique11 DrawTech
{
    pass P0
    {
        SetVertexShader(   CompileShader( vs_5_0, DrawVS() ) );
        SetGeometryShader( CompileShader( gs_5_0, DrawGS() ) );
        SetPixelShader(    CompileShader( ps_5_0, DrawPS() ) );
        
        SetDepthStencilState( NoDepthWrites, 0 );
    }
}